// $Id: CJPEGImage.cpp,v 1.2 2007/02/08 21:07:54 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CJPEGImage.hpp"
#include <Exceptions/CException.hpp>
using Exponent::GUI::Graphics::CJPEGImage;
using Exponent::Exceptions::CException;
#ifdef WIN32
#include <gdiplus.h>
using namespace Gdiplus;
#endif

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CJPEGImage, CBaseImage);

//	===========================================================================
CJPEGImage::CJPEGImage(const CSystemString &filename) 
{
	m_hasAlpha = false;
	EXPONENT_CLASS_CONSTRUCTION(CJPEGImage);

	if (!this->loadFromFile(filename))
	{
		CString message = "Failed to load ";
		message.appendString(filename.getString());
		message.appendString("\nThe reason was : ");
		message.appendString(m_lastError);
		throw CException(message, "CJPEGImage::CJPEGImage(const CSystemString &)");
	}
}

//	===========================================================================
CJPEGImage::CJPEGImage()
{
	m_hasAlpha = false;
	EXPONENT_CLASS_CONSTRUCTION(CJPEGImage);
}

//	===========================================================================
CJPEGImage::~CJPEGImage()
{
	EXPONENT_CLASS_DESTRUCTION(CJPEGImage);
}

//	===========================================================================
bool CJPEGImage::loadFromFile(const CSystemString &filename)
{
	m_filename = filename;
	if (m_filename.hasExtension("jpg") || m_filename.hasExtension("jpeg"))
	{
		// Unload the old image
		if (this->hasImageLoaded())
		{
			this->unloadImage();
		}

	#ifdef WIN32	
		ULONG_PTR gdiPlusToken;		
		// Now we try to attatch to the GDI+ startup process
		GdiplusStartupInput startupInput;
		bool isValid = false;
		if (GdiplusStartup(&gdiPlusToken, &startupInput, NULL) != Ok)
		{
			m_lastError = "System doesnt support GDI+ which must be available to load JPEG images";
			return false;
		}
		else
		{
			// Convert the string to WCHAR
			const size_t length = strlen(filename.getString()) + 1;
			wchar_t *theText    = new wchar_t[length];
			memset(theText, 0, length * sizeof(wchar_t));
			MultiByteToWideChar(CP_ACP, NULL, filename.getString(), -1, theText, (int)length);

			Gdiplus::Bitmap myImage(theText, FALSE);
			isValid = (myImage.GetHBITMAP(NULL, &m_theBitmap) == Ok);
			FREE_ARRAY_POINTER(theText);

			if (isValid)
			{
				// Set the size and the dimensions
				m_size.setRect(0, 0, (long)myImage.GetWidth(), (long)myImage.GetWidth());
				m_dimension.setDimension((long)myImage.GetWidth(), (long)myImage.GetWidth());
				m_imageLoaded = true;
			}
			else
			{
				m_imageLoaded = false;
				m_lastError   = "Failed to aquire HBITMAP from Bitmap image";
			}
		}

		// Shutdown GDI+
		GdiplusShutdown(gdiPlusToken);

		return isValid;
	#else
		
		CSystemString myPath = filename;
		CFStringRef filename = myPath.getStringAsCFStringRef();
		CFURLRef url		 = CFURLCreateWithFileSystemPath(kCFAllocatorDefault, filename, kCFURLPOSIXPathStyle, false);
		CFRelease(filename);
		
		CGDataProviderRef provider = CGDataProviderCreateWithURL(url);
		CFRelease (url);

		m_theBitmap = CGImageCreateWithJPEGDataProvider(provider, NULL, true, kCGRenderingIntentDefault);
		CGDataProviderRelease(provider);
		
		if (m_theBitmap == NULL)
		{
			m_lastError = "Failed to load image with JPEG Data provider";
			m_imageLoaded = false;
			return false;
		}
		else
		{
			const long height = (long)CGImageGetHeight(m_theBitmap);
			const long width  = (long)CGImageGetWidth(m_theBitmap);
			m_size.setRect(0, 0, width, height);
			m_dimension.setDimension(width, height);
			m_imageLoaded = true;
			return true;
		}

	#endif
	}
	else
	{
		m_lastError = "Filename is not of JPEG type";
	}
	return false;
}

//	===========================================================================
CJPEGImage *CJPEGImage::getNewInstance(const CSystemString &path, const CString &filename)
{
	// Construct the proper path
	CSystemString theFilename = path;
	theFilename.appendPath(filename);

	// Return the new image
	return new CJPEGImage(theFilename);
}